/* Distributed under the Apache License, Version 2.0.
   See accompanying NOTICE file for details.*/

#include <iostream>
#include <fstream>
#include "LBM.h"
#include "FileUtils.h"
#ifdef VTK_EXTENSION
  #include "LBMVTK.h"
#endif
#ifdef ITK_EXTENSION
  #include "LBMITK.h"
#endif

enum class RunMode {IMAGE, RUNS};
int main(int argc, char* argv[])
{
  // Specify which CUDA device to use, or leave black to automatically choose the largest device found.
  if (!cuda_device_check())
  {
    std::cerr << "CUDA device check failed, ensure you have specified a valid CUDA device id.\n";
    return 1;
  }

  // Where should we look for input files?
  std::string line;
  std::string test_dir;
  std::ifstream run_config;
  run_config.open("run.config");
  while (!run_config.eof()) // To get you all the lines.
  {
    std::getline(run_config, line); // Get the line.
    size_t pos = line.find("test_dir=");
    if (pos != std::string::npos)
    {
      test_dir = line.substr(9);
      break;
    }
  }
  run_config.close();

  // What do you want to do?
  RunMode mode = RunMode::IMAGE;

  switch (mode)
  {
  case RunMode::IMAGE:
  {
#ifdef ITK_EXTENSION
  // Run LBM via ITK reading in a nrrd file with labels
    try
    {
      std::vector<std::string> nrrd_files;
      ListFiles(test_dir + "/images", nrrd_files, ".nrrd");

      for (std::string nrrd_file : nrrd_files)
      {
        LBM lbm;
        lbm.cfg.reference_pressure = 101300.f;
        lbm.cfg.speed_of_sound = 340.f;
        lbm.cfg.viscosity = 1.f;
        lbm.cfg.density = 1.225f;
        lbm.cfg.thermal_coefficient = 1.f;
        lbm.cfg.wall_temperature = 32.6f;
        lbm.cfg.fluid_temperature = 20.f;
        // Impose a pressure drop
        lbm.cfg.inlet_boundary_condition = LBM::BoundaryCondition::Pressure;
        lbm.cfg.inlet_value = 0;
        lbm.cfg.outlet_boundary_condition = LBM::BoundaryCondition::Pressure;
        lbm.cfg.outlet_value = -10; // negative pulet pressure to yield a positive DelP for Poiseuille's law
        lbm.cfg.downsample_fraction = 0.5;
        lbm.cfg.expanded_results = true;
        //lbm.cfg.setup_only = true; // Don't run, just setup our structures

        // Provide a label mapping
        lbm.in.source_to_lbm_label_map[0] = LBM::BoundaryTypes::Inactive;
        lbm.in.source_to_lbm_label_map[1] = LBM::BoundaryTypes::Wall;   // Left Wall
        lbm.in.source_to_lbm_label_map[2] = LBM::BoundaryTypes::Wall;   // Right Wall
        lbm.in.source_to_lbm_label_map[3] = LBM::BoundaryTypes::Wall;   // Left Outlet Wall
        lbm.in.source_to_lbm_label_map[4] = LBM::BoundaryTypes::Wall;   // Right Outlet Wall
        lbm.in.source_to_lbm_label_map[5] = LBM::BoundaryTypes::Wall;   // Wall Behind Split
        lbm.in.source_to_lbm_label_map[11] = LBM::BoundaryTypes::Inlet; // Left Inlet
        lbm.in.source_to_lbm_label_map[12] = LBM::BoundaryTypes::Inlet; // Right Inlet
        lbm.in.source_to_lbm_label_map[20] = LBM::BoundaryTypes::Outlet;
        if (!RunLBMITK(nrrd_file, lbm))
        {
          std::cerr << "Unable to run LBM on file : " + nrrd_file + "\n";
          return 1;
        }
  #ifdef VTK_EXTENSION
        std::string dir = nrrd_file.substr(0, nrrd_file.find_last_of("/"));
        // Make the output directory
        std::string out_dir = "./test_runs/" + dir.substr(test_dir.length());
        CreateFilePath(out_dir);

        std::string output = nrrd_file.substr(nrrd_file.find_last_of("/"));
        output = Replace(output, ".nrrd", ".vtr");
        std::cout << "Writing out VTR file " + out_dir+output + "\n";
        LBMVTK::WriteVTR(out_dir+output, lbm);
  #endif
      }
    }
    catch (std::exception ex)
    {
      std::cerr << ex.what() << std::endl;
      return 1;
    }
#else
    std::cout << "Enable ITK to run in Image mode\n";
#endif
    break;
  }
  case RunMode::RUNS:
  {
    try
    {
      std::vector<std::string> input_paths;
      std::vector<std::string> output_paths;
      GetRunDirectories(test_dir + "/runs", input_paths, output_paths);
      for (size_t i = 0; i < input_paths.size(); i++)
      {
        LBM lbm;
        std::string input_path = input_paths[i];
        std::string output_path = output_paths[i];
        if (!LBM::LoadLegacyScenario(input_path + "setprob.data",
                                     input_path + "voxelgrid.data",
                                     input_path + "grid.data",
                                     lbm))
        {
          std::cerr << "Unable to load configuration file.\n";
          return 1;
        }
        // Request expanded results
        lbm.cfg.expanded_results = true;
        // Set up our geometry
        if (!lbm.Run())
        {
          std::cerr << "Error running LBM.\n";
          return 1;
        }
#ifdef VTK_EXTENSION
        std::string vtr_out = output_path + "/out.vtr";
        std::cout << "Writing out VTR file : " << vtr_out << "\n";
        LBMVTK::WriteVTR(vtr_out, lbm);
        // Write and read back in the vtr file
        //std::cout << "Testing I/O functionality...\n";
        //LBM lbm2;
        //LBMVTK::ReadVTR("./out.vtr", lbm2);
        //// Compare what we read in to what we ran
        //if (!LBM::CompareDataSet(lbm, lbm2, true))
        //  std::cerr << "The same LBM result does not match!?!";
        //else
        //  std::cout << "Comparison successfull!!";
#endif
      }
    }
    catch (std::exception ex)
    {
      std::cerr << ex.what() << std::endl;
      return 1;
    }

  }
  }

  return 0;
}


