cmake_minimum_required(VERSION 3.30)

include(${CMAKE_CURRENT_LIST_DIR}/json.cmake)
include(${CMAKE_CURRENT_LIST_DIR}/verify-snippet.cmake)
# Test CALLBACK script. Prints output information and verifies index file
# Called as: cmake -P hook.cmake [CheckForStaticQuery?] [index.json]
set(index ${CMAKE_ARGV4})
if (NOT ${CMAKE_ARGV3})
  set(hasStaticInfo "UNEXPECTED")
endif()
read_json("${index}" contents)
string(JSON hook GET "${contents}" hook)

# Output is verified by *-stdout.txt files that the HOOK is run
message(STATUS ${hook})
# Not a check-*.cmake script, this is called as an instrumentation CALLBACK
set(ERROR_MESSAGE "")
function(add_error error)
  string(APPEND ERROR_MESSAGE "${error}\n")
  return(PROPAGATE ERROR_MESSAGE)
endfunction()

function(has_key_index key json)
  cmake_parse_arguments(ARG "UNEXPECTED" "" "" ${ARGN})
  unset(missingKey)
  string(JSON ${key} ERROR_VARIABLE missingKey GET "${json}" ${key})
  if (NOT ARG_UNEXPECTED AND NOT missingKey MATCHES NOTFOUND)
    add_error("\nKey \"${key}\" not in index:\n${json}")
  elseif(ARG_UNEXPECTED AND missingKey MATCHES NOTFOUND)
    add_error("\nUnexpected key \"${key}\" in index:\n${json}")
  endif()
  return(PROPAGATE ERROR_MESSAGE ${key})
endfunction()

has_key_index(version "${contents}")
has_key_index(buildDir "${contents}")
has_key_index(dataDir "${contents}")
has_key_index(snippets "${contents}")

if (NOT version EQUAL 1)
    add_error("Version must be 1, got: ${version}")
endif()

string(JSON length LENGTH "${snippets}")
math(EXPR length "${length}-1")
foreach(i RANGE ${length})
  string(JSON filename GET "${snippets}" ${i})
  if (NOT EXISTS ${dataDir}/${filename})
    add_error("Listed snippet: ${dataDir}/${filename} does not exist")
  endif()
  read_json(${dataDir}/${filename} snippet_contents)
  verify_snippet(${dataDir}/${filename} "${snippet_contents}")
endforeach()

has_key_index(staticSystemInformation "${contents}" ${hasStaticInfo})
if (NOT hasStaticInfo STREQUAL UNEXPECTED)
  has_key_index(OSName "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(OSPlatform "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(OSRelease "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(OSVersion "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(familyId "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(hostname "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(is64Bits "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(modelId "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(numberOfLogicalCPU "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(numberOfPhysicalCPU "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(processorAPICID "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(processorCacheSize "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(processorClockFrequency "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(processorName "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(totalPhysicalMemory "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(totalVirtualMemory "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(vendorID "${staticSystemInformation}" ${hasStaticInfo})
  has_key_index(vendorString "${staticSystemInformation}" ${hasStaticInfo})
endif()

get_filename_component(dataDir ${index} DIRECTORY)
get_filename_component(v1 ${dataDir} DIRECTORY)
if (EXISTS ${v1}/${hook}.hook)
  add_error("Received multiple triggers of the same hook: ${hook}")
endif()
file(WRITE ${v1}/${hook}.hook "${ERROR_MESSAGE}")

if (NOT ERROR_MESSAGE MATCHES "^$")
  message(FATAL_ERROR ${ERROR_MESSAGE})
endif()
