#ifndef VITAL_DESCRIPTOR_ARRAY_OF_H_
#define VITAL_DESCRIPTOR_ARRAY_OF_H_


#include "descriptor.hpp"

// ------------------------------------------------------------------
/// Abstract base class of a descriptor containing an array of type T
template < typename T >
class descriptor_array_of :
  public descriptor
{
public:
  /// Access the type info of the underlying data (double or float)
  virtual std::type_info const& data_type() const { return typeid( T ); }

  /// The number of bytes used to represent the data
  std::size_t num_bytes() const { return this->size() * sizeof( T ); }


  /// Return the descriptor as a vector of doubles
  std::vector< double > as_double() const
  {
    const std::size_t length = this->size();
    std::vector< double > double_data( length );

    for ( std::size_t i = 0; i < length; ++i )
    {
      double_data[i] = static_cast< double > ( this->raw_data()[i] );
    }
    return double_data;
  }

  virtual const byte* as_bytes() const
  {
    return reinterpret_cast<const byte *>(raw_data());
  }

  /// Return an pointer to the raw data array
  virtual T* raw_data() = 0;

  /// Return an pointer to the raw data array
  virtual const T* raw_data() const = 0;

  // Iterator interface
  T const* begin() const { return this->raw_data(); }
  T const* end() const { return this->raw_data() + this->size(); }


  /// Equality operator
  bool operator==( descriptor_array_of<T> const& other ) const
  {
    if( this->size() != other.size() )
    {
      return false;
    }
    return std::equal(this->raw_data(), this->raw_data() + this->size(),
                      other.raw_data());
  }

  /// Inequality operator
  bool operator!=( descriptor_array_of<T> const& other ) const
  {
    return ! operator==(other);
  }

};

#endif