/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkInformationVector.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkInformationVector - Store zero or more vtkInformation instances.
// .SECTION Description

// vtkInformationVector stores a vector of zero or more vtkInformation
// objects corresponding to the input or output information for a
// vtkAlgorithm.  An instance of this class is passed to
// vtkAlgorithm::ProcessRequest calls.

#ifndef __vtkInformationVector_h
#define __vtkInformationVector_h

#include "vtkObject.h"

class vtkInformation;
class vtkInformationVectorInternals;

class VTK_FILTERING_EXPORT vtkInformationVector : public vtkObject
{
public:
  static vtkInformationVector *New();
  vtkTypeRevisionMacro(vtkInformationVector,vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Get/Set the number of information objects in the vector.  Setting
  // the number to larger than the current number will create empty
  // vtkInformation instances.  Setting the number to smaller than the
  // current number will remove entries from higher indices.
  int GetNumberOfInformationObjects();
  void SetNumberOfInformationObjects(int n);

  // Description:
  // Get/Set the vtkInformation instance stored at the given index in
  // the vector.  The vector will automatically expand to include the
  // index given if necessary.  Missing entries in-between will be
  // filled with empty vtkInformation instances.
  void SetInformationObject(int index, vtkInformation* info);
  vtkInformation* GetInformationObject(int index);

  // Description:
  // Append/Remove an information object.
  void Append(vtkInformation* info);
  void Remove(vtkInformation* info);

  // Description:
  // Initiate garbage collection when a reference is removed.
  virtual void Register(vtkObjectBase* o);
  virtual void UnRegister(vtkObjectBase* o);

protected:
  vtkInformationVector();
  ~vtkInformationVector();

  // Internal implementation details.
  vtkInformationVectorInternals* Internal;

  // Garbage collection support.
  virtual void ReportReferences(vtkGarbageCollector*);
private:
  vtkInformationVector(const vtkInformationVector&);  // Not implemented.
  void operator=(const vtkInformationVector&);  // Not implemented.
};

#endif
