// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use impl_prelude::*;

#[derive(Debug, Default, Clone, Copy)]
/// Rejects the addition of symlinks to a repository.
pub struct RejectSymlinks;

impl RejectSymlinks {
    /// Create a new check which rejects the addition of symlinks.
    pub fn new() -> Self {
        RejectSymlinks
    }
}

impl ContentCheck for RejectSymlinks {
    fn name(&self) -> &str {
        "reject-symlinks"
    }

    fn check(&self, _: &CheckGitContext, content: &Content) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        for diff in content.diffs() {
            match diff.status {
                StatusChange::Added |
                StatusChange::Modified(_) => (),
                _ => continue,
            }

            if diff.new_mode == "120000" {
                result.add_error(format!("{}adds a symlink at `{}` which is not allowed.",
                                         commit_prefix(content),
                                         diff.name));
            }
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use checks::RejectSymlinks;
    use checks::test::*;

    static BAD_TOPIC: &'static str = "00ffdf352196c16a453970de022a8b4343610ccf";
    static FIX_TOPIC: &'static str = "d93ffc2e8b782ba8dce2278dd86fda0df80f454b";

    #[test]
    fn test_reject_symlinks() {
        let check = RejectSymlinks;
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_reject_symlinks", BAD_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 4);
        assert_eq!(result.errors()[0],
                   "commit 00ffdf352196c16a453970de022a8b4343610ccf adds a symlink at \
                    `absolute-link` which is not allowed.");
        assert_eq!(result.errors()[1],
                   "commit 00ffdf352196c16a453970de022a8b4343610ccf adds a symlink at \
                    `broken-link` which is not allowed.");
        assert_eq!(result.errors()[2],
                   "commit 00ffdf352196c16a453970de022a8b4343610ccf adds a symlink at \
                    `inside-link` which is not allowed.");
        assert_eq!(result.errors()[3],
                   "commit 00ffdf352196c16a453970de022a8b4343610ccf adds a symlink at \
                    `outside-link` which is not allowed.");
        assert_eq!(result.temporary(), false);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }

    #[test]
    fn test_reject_symlinks_topic() {
        let check = RejectSymlinks;
        let mut conf = GitCheckConfiguration::new();

        conf.add_topic_check(&check);

        let result = test_check("test_reject_symlinks_topic", BAD_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 4);
        assert_eq!(result.errors()[0],
                   "adds a symlink at `absolute-link` which is not allowed.");
        assert_eq!(result.errors()[1],
                   "adds a symlink at `broken-link` which is not allowed.");
        assert_eq!(result.errors()[2],
                   "adds a symlink at `inside-link` which is not allowed.");
        assert_eq!(result.errors()[3],
                   "adds a symlink at `outside-link` which is not allowed.");
        assert_eq!(result.temporary(), false);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }

    #[test]
    fn test_reject_symlinks_topic_fixed() {
        let check = RejectSymlinks;
        let mut conf = GitCheckConfiguration::new();

        conf.add_topic_check(&check);

        let result = test_check("test_reject_symlinks_topic_fixed", FIX_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 0);
        assert_eq!(result.temporary(), false);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), true);
    }
}
