// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use impl_prelude::*;

#[derive(Debug)]
/// A check which denies commits which modify files underneath certain path.
pub struct RestrictedPath {
    /// The path which may not be edited.
    path: String,
    /// Whether the check is an error or a warning.
    required: bool,
}

impl RestrictedPath {
    /// Create a check which rejects modifications to files underneath a of given path.
    pub fn new<S: ToString>(path: S) -> Self {
        RestrictedPath {
            path: path.to_string(),
            required: true,
        }
    }

    /// Whether commits changing the path are errors or warnings.
    pub fn required(&mut self, required: bool) -> &mut Self {
        self.required = required;
        self
    }
}

impl Check for RestrictedPath {
    fn name(&self) -> &str {
        "restricted-path"
    }

    fn check(&self, _: &CheckGitContext, commit: &Commit) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        let is_restricted = commit.diffs
            .iter()
            .map(|diff| diff.name.as_path())
            .any(|path| path.starts_with(&self.path));

        if is_restricted {
            if self.required {
                result.add_error(format!("commit {} not allowed; the `{}` path is restricted.",
                                         commit.sha1,
                                         self.path));
            } else {
                result.add_warning(format!("commit {} should be inspected; the `{}` path is \
                                            restricted.",
                                           commit.sha1,
                                           self.path));
            };
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use checks::RestrictedPath;
    use checks::test::*;

    static BAD_TOPIC: &'static str = "e845fa2521c17bdd31d5891c1c644fb17f0629db";

    #[test]
    fn test_restricted_path() {
        let check = RestrictedPath::new("restricted");
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_restricted_path", BAD_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 1);
        assert_eq!(result.errors()[0],
                   "commit e845fa2521c17bdd31d5891c1c644fb17f0629db not allowed; the \
                    `restricted` path is restricted.");
        assert_eq!(result.temporary(), false);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }

    #[test]
    fn test_restricted_path_warning() {
        let mut check = RestrictedPath::new("restricted");
        let mut conf = GitCheckConfiguration::new();

        check.required(false);

        conf.add_check(&check);

        let result = test_check("test_restricted_path_warning", BAD_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 1);
        assert_eq!(result.warnings()[0],
                   "commit e845fa2521c17bdd31d5891c1c644fb17f0629db should be inspected; the \
                    `restricted` path is restricted.");
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 0);
        assert_eq!(result.temporary(), false);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), true);
    }
}
