// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use super::super::*;

#[derive(Debug)]
/// A check which denies commits which modify files underneath certain path.
pub struct RestrictedPath {
    path: String,
}

impl RestrictedPath {
    /// Create a check which rejects modifications to files underneath a of given path.
    pub fn new<S: ToString>(path: S) -> Self {
        RestrictedPath {
            path: path.to_string(),
        }
    }
}

impl Check for RestrictedPath {
    fn name(&self) -> &str {
        "restricted-path"
    }

    fn check(&self, _: &CheckGitContext, commit: &Commit) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        let is_restricted = commit.diffs
            .iter()
            .map(|diff| diff.name.as_path())
            .any(|path| path.starts_with(&self.path));

        if is_restricted {
            result.add_error(format!("commit {} not allowed; the `{}` path is restricted.",
                                     commit.sha1_short,
                                     self.path));
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use super::RestrictedPath;
    use super::super::test::*;

    static BAD_TOPIC: &'static str = "e845fa2521c17bdd31d5891c1c644fb17f0629db";

    #[test]
    fn test_restricted_path() {
        let check = RestrictedPath::new("restricted");
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_restricted_path", BAD_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 1);
        assert_eq!(result.errors()[0],
                   "commit e845fa2 not allowed; the `restricted` path is restricted.");
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }
}
