// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use super::super::*;

#[derive(Debug, Default, Clone, Copy)]
/// Rejects the addition of symlinks to a repository.
pub struct RejectSymlinks;

impl RejectSymlinks {
    /// Create a new check which rejects the addition of symlinks.
    pub fn new() -> Self {
        RejectSymlinks
    }
}

impl Check for RejectSymlinks {
    fn name(&self) -> &str {
        "reject-symlinks"
    }

    fn check(&self, _: &CheckGitContext, commit: &Commit) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        for diff in &commit.diffs {
            match diff.status {
                StatusChange::Added |
                StatusChange::Modified(_) => (),
                _ => continue,
            }

            if diff.new_mode == "120000" {
                result.add_error(format!("commit {} adds a symlink at `{}` which is not allowed.",
                                         commit.sha1_short,
                                         diff.name));
            }
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use super::RejectSymlinks;
    use super::super::test::*;

    static BAD_TOPIC: &'static str = "00ffdf352196c16a453970de022a8b4343610ccf";

    #[test]
    fn test_reject_symlinks() {
        let check = RejectSymlinks;
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_reject_symlinks", BAD_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 4);
        assert_eq!(result.errors()[0],
                   "commit 00ffdf3 adds a symlink at `absolute-link` which is not allowed.");
        assert_eq!(result.errors()[1],
                   "commit 00ffdf3 adds a symlink at `broken-link` which is not allowed.");
        assert_eq!(result.errors()[2],
                   "commit 00ffdf3 adds a symlink at `inside-link` which is not allowed.");
        assert_eq!(result.errors()[3],
                   "commit 00ffdf3 adds a symlink at `outside-link` which is not allowed.");
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }
}
