// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use super::super::*;

#[derive(Debug, Default, Clone, Copy)]
/// A check which denies root commits.
pub struct RejectSeparateRoot;

impl RejectSeparateRoot {
    /// Create a new check to reject new root commits.
    pub fn new() -> Self {
        RejectSeparateRoot {}
    }
}

impl Check for RejectSeparateRoot {
    fn name(&self) -> &str {
        "reject-separate-root"
    }

    fn check(&self, _: &CheckGitContext, commit: &Commit) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        if commit.parents.is_empty() {
            result.add_error(format!("commit {} not allowed; it is a root commit.",
                                     commit.sha1_short));
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use super::RejectSeparateRoot;
    use super::super::test::*;

    static NO_ROOT_TOPIC: &'static str = "ba3dc3cb09a558c88282742413a2dccb17d444fc";
    static WITH_ROOT_TOPIC: &'static str = "ff560e8798ef7a9d10bf43660695f7155b49b398";

    #[test]
    fn test_reject_separate_root_no_root() {
        let check = RejectSeparateRoot::new();
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_reject_separate_root_no_root", NO_ROOT_TOPIC, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 0);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), true);
    }

    #[test]
    fn test_reject_separate_root_with_root() {
        let check = RejectSeparateRoot::new();
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_reject_separate_root_with_root",
                                WITH_ROOT_TOPIC,
                                &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 1);
        assert_eq!(result.errors()[0],
                   "commit ff560e8 not allowed; it is a root commit.");
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }
}
