// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use super::super::*;

#[derive(Debug, Default, Clone, Copy)]
/// Check for files which lack an end-of-line at the end of the file.
pub struct CheckEndOfLine;

impl CheckEndOfLine {
    /// Create a new check to check for end-of-lines at the end of files.
    pub fn new() -> Self {
        CheckEndOfLine
    }
}

impl Check for CheckEndOfLine {
    fn name(&self) -> &str {
        "check-end-of-line"
    }

    fn check(&self, _: &CheckGitContext, commit: &Commit) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        for diff in &commit.diffs {
            match diff.status {
                StatusChange::Added |
                StatusChange::Modified(_) => (),
                _ => continue,
            }

            // Ignore symlinks; they only end with newlines if they point to a file with a newline
            // at the end of its name.
            if diff.new_mode == "120000" {
                continue;
            }

            // FIXME: does this work for binary files?
            let patch = match commit.file_patch(&diff.name) {
                Ok(s) => s,
                Err(err) => {
                    result.add_alert(format!("failed to get the diff for file `{}` in commit \
                                              {}: {}",
                                             diff.name,
                                             commit.sha1_short,
                                             err),
                                     true);
                    continue;
                },
            };

            let has_missing_newline = patch.lines()
                .any(|line| line == "\\ No newline at end of file");

            if has_missing_newline {
                result.add_error(format!("commit {} is not allowed; missing newline at the end \
                                          of file in `{}`.",
                                         commit.sha1_short,
                                         diff.name));
            }
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use super::CheckEndOfLine;
    use super::super::test::*;

    static BAD_COMMIT: &'static str = "829cdf8cb069b8f8a634a034d3f85089271601cf";

    #[test]
    fn test_check_end_of_line() {
        let check = CheckEndOfLine::new();
        let mut conf = GitCheckConfiguration::new();

        conf.add_check(&check);

        let result = test_check("test_check_end_of_line", BAD_COMMIT, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 1);
        assert_eq!(result.errors()[0],
                   "commit 829cdf8 is not allowed; missing newline at the end of file in \
                    `missing-newline-eof`.");
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), false);
    }
}
