// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate git_workarea;
use self::git_workarea::CommitId;

use super::super::*;

#[derive(Debug)]
/// A check to allow robots to skip all checks.
pub struct AllowRobot {
    identity: Identity,
}

impl AllowRobot {
    /// Create a new check to allow robots to skip checks.
    ///
    /// Any actions performed by the identity are waved through and all other checks are ignored.
    pub fn new(identity: Identity) -> Self {
        AllowRobot {
            identity: identity,
        }
    }
}

impl BranchCheck for AllowRobot {
    fn name(&self) -> &str {
        "allow-robot"
    }

    fn check(&self, ctx: &CheckGitContext, _: &CommitId) -> Result<CheckResult> {
        let mut result = CheckResult::new();

        if *ctx.topic_owner() == self.identity {
            result.whitelist();
        }

        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use super::AllowRobot;
    use super::super::test::*;

    static ALLOW_ROBOT_COMMIT: &'static str = "43adb8173eb6d7a39f98e1ec3351cf27414c9aa1";

    #[test]
    fn test_allow_robot_allowed() {
        let check = AllowRobot::new(Identity::new("Rust Git Checks Tests",
                                                  "rust-git-checks@example.com"));
        let mut conf = GitCheckConfiguration::new();

        conf.add_branch_check(&check);

        let result = test_check("test_allow_robot_allowed", ALLOW_ROBOT_COMMIT, &conf);

        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 0);
        assert_eq!(result.allowed(), true);
        assert_eq!(result.pass(), true);
    }

    #[test]
    fn test_allow_robot_not_robot_not_whitelisted() {
        let check = AllowRobot::new(Identity::new("Rust Git Checks 7ests",
                                                  "rust-git-checks@example.com"));
        let mut conf = GitCheckConfiguration::new();

        conf.add_branch_check(&check);

        let result = test_check("test_allow_robot_not_robot_not_whitelisted",
                                ALLOW_ROBOT_COMMIT,
                                &conf);


        assert_eq!(result.warnings().len(), 0);
        assert_eq!(result.alerts().len(), 0);
        assert_eq!(result.errors().len(), 0);
        assert_eq!(result.allowed(), false);
        assert_eq!(result.pass(), true);
    }
}
