// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate git_workarea;
use self::git_workarea::CommitId;

use super::commit::Commit;
use super::context::CheckGitContext;
use super::error::Result;

#[derive(Debug, Default)]
/// The results of a check.
pub struct CheckResult {
    warnings: Vec<String>,
    alerts: Vec<String>,
    errors: Vec<String>,

    allow: bool,
    pass: bool,
}

impl CheckResult {
    /// Create a new results structure.
    pub fn new() -> Self {
        CheckResult {
            warnings: vec![],
            alerts: vec![],
            errors: vec![],

            allow: false,
            pass: true,
        }
    }

    /// Adds a warning message to the results.
    pub fn add_warning<S: ToString>(&mut self, warning: S) -> &mut Self {
        self.warnings.push(warning.to_string());

        self
    }

    /// Adds an alert to the results.
    ///
    /// These messages should be brought to the attention of those maintaining the deployment of
    /// the checks.
    pub fn add_alert<S: ToString>(&mut self, alert: S, should_block: bool) -> &mut Self {
        self.alerts.push(alert.to_string());
        self.pass = self.pass && !should_block;

        self
    }

    /// Adds a error message to the results.
    ///
    /// Also marks the checks as having failed.
    pub fn add_error<S: ToString>(&mut self, error: S) -> &mut Self {
        self.errors.push(error.to_string());
        self.pass = false;

        self
    }

    /// Allows the checks to pass no matter what.
    pub fn whitelist(&mut self) -> &mut Self {
        self.allow = true;

        self
    }

    /// The warnings from the checks.
    pub fn warnings(&self) -> &Vec<String> {
        &self.warnings
    }

    /// The alerts from the checks.
    pub fn alerts(&self) -> &Vec<String> {
        &self.alerts
    }

    /// The errors from the checks.
    pub fn errors(&self) -> &Vec<String> {
        &self.errors
    }

    /// Whether the checks will allow the commit no matter what.
    pub fn allowed(&self) -> bool {
        self.allow
    }

    /// Whether the checks passed or failed.
    pub fn pass(&self) -> bool {
        self.pass
    }

    /// Combine two results together.
    pub fn combine(self, other: Self) -> Self {
        CheckResult {
            warnings: self.warnings.into_iter().chain(other.warnings.into_iter()).collect(),
            alerts: self.alerts.into_iter().chain(other.alerts.into_iter()).collect(),
            errors: self.errors.into_iter().chain(other.errors.into_iter()).collect(),

            allow: self.allow || other.allow,
            pass: self.pass && other.pass,
        }
    }
}

/// Interface for checks which run for each commit.
pub trait Check: Sync {
    /// The name of the check.
    fn name(&self) -> &str;

    /// Run the check.
    fn check(&self, ctx: &CheckGitContext, commit: &Commit) -> Result<CheckResult>;
}

/// Interface for checks which runs once for the entire branch.
pub trait BranchCheck: Sync {
    /// The name of the check.
    fn name(&self) -> &str;

    /// Run the check.
    fn check(&self, ctx: &CheckGitContext, commit: &CommitId) -> Result<CheckResult>;
}
