// When creating shaders for Universal Render Pipeline you can you the ShaderGraph which is super AWESOME!
// However, if you want to author shaders in shading language you can use this teamplate as a base.
// Please note, this shader does not necessarily match perfomance of the built-in URP Lit shader.
// This shader works with URP 7.1.x and above
// HS-2022/11/22 Based on https://gist.github.com/phi-lira/225cd7c5e8545be602dca4eb5ed111ba 
Shader "Universal Render Pipeline/iMSTK/Connective Tissue-v2"
{
    // Specular Highlights still look weird, alpha blending is not quite working right either
    // 


    Properties
    {
        // These are parameters for the geometry shader
        // The thickness of the line in screenspace
        _Thickness("Thickness", Range(0.0, 10.0)) = 1.0

        // The thickness of the UV strip with the color texture
        _UVThickness("UV Thickness", Range(0.0, 1.0)) = .8

        // Determines visual curvature of the strip, smaller more 
        // curvature, bigger less curvature
        _CenterOffset("Center Offset Factor", Range(0.0001, 10.0)) = 1.0
        [NoScaleOffset] _AlphaMap("Alpha Map", 2D) = "white" {}

        // Specular vs Metallic workflow
        [HideInInspector] _WorkflowMode("WorkflowMode", Float) = 1.0

        [MainColor] _BaseColor("Color", Color) = (0.5,0.5,0.5,1)
        [MainTexture] _BaseMap("Albedo", 2D) = "white" {}

        _Cutoff("Alpha Cutoff", Range(0.0, 1.0)) = 0.5

        _Smoothness("Smoothness", Range(0.0, 1.0)) = 0.5
        _GlossMapScale("Smoothness Scale", Range(0.0, 1.0)) = 1.0
        _SmoothnessTextureChannel("Smoothness texture channel", Float) = 0

        [Gamma] _Metallic("Metallic", Range(0.0, 1.0)) = 0.0
        _MetallicGlossMap("Metallic", 2D) = "white" {}

        _SpecColor("Specular", Color) = (0.2, 0.2, 0.2)
        _SpecGlossMap("Specular", 2D) = "white" {}

        [ToggleOff] _SpecularHighlights("Specular Highlights", Float) = 1.0
        [ToggleOff] _EnvironmentReflections("Environment Reflections", Float) = 1.0

        _BumpScale("Scale", Float) = 1.0
        _BumpMap("Normal Map", 2D) = "bump" {}

        _OcclusionStrength("Strength", Range(0.0, 1.0)) = 1.0
        _OcclusionMap("Occlusion", 2D) = "white" {}

        _EmissionColor("Color", Color) = (0,0,0)
        _EmissionMap("Emission", 2D) = "white" {}

        // Blending state
        [HideInInspector] _Surface("__surface", Float) = 0.0
        [HideInInspector] _Blend("__blend", Float) = 0.0
        [HideInInspector] _AlphaClip("__clip", Float) = 0.0
        [HideInInspector] _SrcBlend("__src", Float) = 1.0
        [HideInInspector] _DstBlend("__dst", Float) = 0.0
        [HideInInspector] _ZWrite("__zw", Float) = 1.0
        [HideInInspector] _Cull("__cull", Float) = 2.0

        _ReceiveShadows("Receive Shadows", Float) = 1.0

        // Editmode props
        [HideInInspector] _QueueOffset("Queue offset", Float) = 0.0
    }

        SubShader
        {
            // With SRP we introduce a new "RenderPipeline" tag in Subshader. This allows to create shaders
            // that can match multiple render pipelines. If a RenderPipeline tag is not set it will match
            // any render pipeline. In case you want your subshader to only run in LWRP set the tag to
            // "UniversalRenderPipeline"
            Tags{"RenderType" = "Transparent" "RenderPipeline" = "UniversalRenderPipeline" "IgnoreProjector" = "True"}
            LOD 300

            // ------------------------------------------------------------------
            // Forward pass. Shades GI, emission, fog and all lights in a single pass.
            // Compared to Builtin pipeline forward renderer, LWRP forward renderer will
            // render a scene with multiple lights with less drawcalls and less overdraw.
            Pass
            {
                // "Lightmode" tag must be "UniversalForward" or not be defined in order for
                // to render objects.
                Name "StandardLit"
                //Tags{"LightMode" = "UniversalForward"}

                Blend[_SrcBlend][_DstBlend]
                ZWrite[_ZWrite]
                Cull[_Cull]

                HLSLPROGRAM
            // Required to compile gles 2.0 with standard SRP library
            // All shaders must be compiled with HLSLcc and currently only gles is not using HLSLcc by default
            #pragma prefer_hlslcc gles
            #pragma exclude_renderers d3d11_9x
            #pragma target 2.0

            // -------------------------------------
            // Material Keywords
            // unused shader_feature variants are stripped from build automatically
            #pragma shader_feature _NORMALMAP
            #pragma shader_feature _ALPHATEST_ON
            #pragma shader_feature _ALPHAPREMULTIPLY_ON
            #pragma shader_feature _EMISSION
            #pragma shader_feature _METALLICSPECGLOSSMAP
            #pragma shader_feature _SMOOTHNESS_TEXTURE_ALBEDO_CHANNEL_A
            #pragma shader_feature _OCCLUSIONMAP

            #pragma shader_feature _SPECULARHIGHLIGHTS_OFF
            #pragma shader_feature _GLOSSYREFLECTIONS_OFF
            #pragma shader_feature _SPECULAR_SETUP
            #pragma shader_feature _RECEIVE_SHADOWS_OFF

            // -------------------------------------
            // Universal Render Pipeline keywords
            // When doing custom shaders you most often want to copy and past these #pragmas
            // These multi_compile variants are stripped from the build depending on:
            // 1) Settings in the LWRP Asset assigned in the GraphicsSettings at build time
            // e.g If you disable AdditionalLights in the asset then all _ADDITIONA_LIGHTS variants
            // will be stripped from build
            // 2) Invalid combinations are stripped. e.g variants with _MAIN_LIGHT_SHADOWS_CASCADE
            // but not _MAIN_LIGHT_SHADOWS are invalid and therefore stripped.
            #pragma multi_compile _ _MAIN_LIGHT_SHADOWS
            #pragma multi_compile _ _MAIN_LIGHT_SHADOWS_CASCADE
            #pragma multi_compile _ _ADDITIONAL_LIGHTS_VERTEX _ADDITIONAL_LIGHTS
            #pragma multi_compile _ _ADDITIONAL_LIGHT_SHADOWS
            #pragma multi_compile _ _SHADOWS_SOFT
            #pragma multi_compile _ _MIXED_LIGHTING_SUBTRACTIVE

            // -------------------------------------
            // Unity defined keywords
            #pragma multi_compile _ DIRLIGHTMAP_COMBINED
            #pragma multi_compile _ LIGHTMAP_ON
            #pragma multi_compile_fog

            //--------------------------------------
            // GPU Instancing
            #pragma multi_compile_instancing

            #pragma vertex LitPassVertex
            #pragma geometry GeometryPass
            #pragma fragment LitPassFragment

            // Including the following two function is enought for shading with Universal Pipeline. Everything is included in them.
            // Core.hlsl will include SRP shader library, all constant buffers not related to materials (perobject, percamera, perframe).
            // It also includes matrix/space conversion functions and fog.
            // Lighting.hlsl will include the light functions/data to abstract light constants. You should use GetMainLight and GetLight functions
            // that initialize Light struct. Lighting.hlsl also include GI, Light BDRF functions. It also includes Shadows.

            // Required by all Universal Render Pipeline shaders.
            // It will include Unity built-in shader variables (except the lighting variables)
            // (https://docs.unity3d.com/Manual/SL-UnityShaderVariables.html
            // It will also include many utilitary functions. 
            #include "Packages/com.unity.render-pipelines.universal/ShaderLibrary/Core.hlsl"

            // Include this if you are doing a lit shader. This includes lighting shader variables,
            // lighting and shadow functions
            #include "Packages/com.unity.render-pipelines.universal/ShaderLibrary/Lighting.hlsl"

            // Material shader variables are not defined in SRP or LWRP shader library.
            // This means _BaseColor, _BaseMap, _BaseMap_ST, and all variables in the Properties section of a shader
            // must be defined by the shader itself. If you define all those properties in CBUFFER named
            // UnityPerMaterial, SRP can cache the material properties between frames and reduce significantly the cost
            // of each drawcall.
            // In this case, for sinmplicity LitInput.hlsl is included. This contains the CBUFFER for the material
            // properties defined above. As one can see this is not part of the ShaderLibrary, it specific to the
            // LWRP Lit shader.
            #include "Packages/com.unity.render-pipelines.universal/Shaders/LitInput.hlsl"

            // Custom Uniforms
            CBUFFER_START(UnityPerMaterial)
            float _Thickness;
            float _UVThickness;
            float _CenterOffset;
            CBUFFER_END

            sampler2D _AlphaMap;

            struct Attributes
            {
                float4 positionOS   : POSITION;
                float3 normalOS     : NORMAL;
                float4 tangentOS    : TANGENT;   // Possibly don't need it's normal cross direction in the line shader
                float2 uv           : TEXCOORD0;
                float2 uvLM         : TEXCOORD1;
                UNITY_VERTEX_INPUT_INSTANCE_ID
            };

            struct VertexToGeometry
            {
                float2 uv                       : TEXCOORD0;
                float2 uvLM                     : TEXCOORD1;
                float3 normalOS                 : NORMAL;    // Carry forward to geometry stage for calculating 
                float4 tangentOS                : TANGENT;
                float4 positionOS               : POSITION;
                UNITY_VERTEX_INPUT_INSTANCE_ID
            };

            struct GeometryToFragment
            {
                float2 uv                       : TEXCOORD0;
                float2 uvLM                     : TEXCOORD1;
                float4 positionWSAndFogFactor   : TEXCOORD2; // xyz: positionWS, w: vertex fog factor
                half3  normalWS                 : TEXCOORD3;

#if _NORMALMAP
                half3 tangentWS                 : TEXCOORD4;
                half3 bitangentWS               : TEXCOORD5;
#endif

#ifdef _MAIN_LIGHT_SHADOWS
                float4 shadowCoord              : TEXCOORD6; // compute shadow coord per-vertex for the main light
#endif
                float4 positionCS               : SV_POSITION;
                float2 uvAlpha                  : TEXCOORD7;
            };

            VertexToGeometry LitPassVertex(Attributes input)
            {
                VertexToGeometry output;

                // Forward to geometry shader
                output.positionOS = input.positionOS;
                output.normalOS = input.normalOS;
                output.tangentOS = input.tangentOS;


                // TRANSFORM_TEX is the same as the old shader library.
                output.uv = TRANSFORM_TEX(input.uv, _BaseMap);
                output.uvLM = input.uvLM.xy * unity_LightmapST.xy + unity_LightmapST.zw;

                return output;
            }

            [maxvertexcount(4)]
            void GeometryPass(line VertexToGeometry input[2], inout TriangleStream<GeometryToFragment> output)
            {
                
                GeometryToFragment g[4]; // Array for output points

                VertexPositionInputs vertexInput[2];

                for (int i = 0; i < 2; ++i) {
                    vertexInput[i] = GetVertexPositionInputs(input[i].positionOS.xyz);
                    float fogFactor = ComputeFogFactor(vertexInput[i].positionCS.z);
                    g[i * 2].positionWSAndFogFactor = float4(vertexInput[i].positionWS, fogFactor);
                    g[i * 2 + 1].positionWSAndFogFactor = float4(vertexInput[i].positionWS, fogFactor);

                    #ifdef _MAIN_LIGHT_SHADOWS
                        // shadow coord for the main light is computed in vertex.
                        // If cascades are enabled, LWRP will resolve shadows in screen space
                        // and this coord will be the uv coord of the screen space shadow texture.
                        // Otherwise LWRP will resolve shadows in light space (no depth pre-pass and shadow collect pass)
                        // In this case shadowCoord will be the position in light space.
                        g[i * 2].shadowCoord = GetShadowCoord(vertexInput[i]);
                        g[i * 2 + 1].shadowCoord = GetShadowCoord(vertexInput[i]);
                    #endif

                    g[i * 2].uvLM = input[i].uvLM;
                    g[i * 2 + 1].uvLM = input[i].uvLM;
                }


                // Vertex Generation Happens in clipspace
                float4 p1 = vertexInput[0].positionCS;
                float4 p2 = vertexInput[1].positionCS;

                float2 dir = normalize(p2.xy - p1.xy);
                float2 normal = float2(-dir.y, dir.x);

                //float4 offset1 = float4(normal * p1.w * _Thickness / 2.0f, 0, 0);
                //float4 offset2 = float4(normal * p2.w * _Thickness / 2.0f, 0, 0);
                float4 offset1 = float4(normal * _Thickness / 2.0f, 0, 0);
                float4 offset2 = float4(normal * _Thickness / 2.0f, 0, 0);
                offset1.x *= _ScreenParams.y / _ScreenParams.x;
                offset2.x *= _ScreenParams.y / _ScreenParams.x;

                float4 o1 = p1 + offset1;
                float4 o2 = p1 - offset1;
                float4 o3 = p2 + offset2;
                float4 o4 = p2 - offset2;

                g[0].positionCS = o1;
                g[1].positionCS = o2;
                g[2].positionCS = o3;
                g[3].positionCS = o4;

                g[0].positionCS.z = 1.0;
                g[1].positionCS.z = 1.0;
                g[2].positionCS.z = 1.0;
                g[3].positionCS.z = 1.0;

                // Normal Generation happens in object space transfer to worldspace 
                float3 directionOS = input[1].positionOS.xyz - input[0].positionOS.xyz;
                for (int i = 0; i < 2; ++i)
                {
                    // Assume that we have a quad in Objectspace, take a point 
                    // in the "inside" of our material and calculate the normals 
                    // by connecting that point inside with each corner
                    // The tangents are calculated by the cross product of the normal
                    // and the direction vector of the line
                    float3 origin = input[i].positionOS.xyz - input[i].normalOS;
                    float3 corner = input[i].positionOS.xyz + input[i].tangentOS.xyz;
                    float3 normal = -normalize(corner - origin * _CenterOffset);
                    float4 tangent = float4(normalize(cross(directionOS, normal)),1);

                    VertexNormalInputs vertexNormalInput = GetVertexNormalInputs(
                        normal, tangent);
                    g[i * 2].normalWS = vertexNormalInput.normalWS;
#ifdef _NORMALMAP
                    g[i * 2].tangentWS = vertexNormalInput.tangentWS;
                    g[i * 2].bitangentWS = vertexNormalInput.bitangentWS;
#endif

                    corner = input[i].positionOS.xyz - input[i].tangentOS.xyz;
                    normal = -normalize(corner - origin * _CenterOffset);
                    tangent = tangent = float4(normalize(cross(directionOS, normal)), 1);

                    vertexNormalInput = GetVertexNormalInputs(
                        normal, tangent);
                    g[i * 2 + 1].normalWS = vertexNormalInput.normalWS;
#ifdef _NORMALMAP
                    g[i * 2 + 1].tangentWS = vertexNormalInput.tangentWS;
                    g[i * 2 + 1].bitangentWS = vertexNormalInput.bitangentWS;
#endif
                }

                // UV generation
                float2 uv1 = input[0].uv;
                float2 uv2 = input[1].uv;

                float2 uvDir = normalize(uv2 - uv1);
                float2 uvOffset = float2(-uvDir.y, uvDir.x) * _UVThickness;


                g[0].uv = uv1 + uvOffset;
                g[1].uv = uv1 - uvOffset;
                g[2].uv = uv2 + uvOffset;
                g[3].uv = uv2 - uvOffset;

                g[0].uvAlpha = float2(0.0, 0.0);
                g[1].uvAlpha = float2(0.0, 1.0);
                g[2].uvAlpha = float2(1.0, 0.0);
                g[3].uvAlpha = float2(1.0, 1.0);

                output.Append(g[0]);
                output.Append(g[2]);
                output.Append(g[1]);
                output.Append(g[3]);
            }


            half4 LitPassFragment(GeometryToFragment input) : SV_Target
            {
                // Surface data contains albedo, metallic, specular, smoothness, occlusion, emission and alpha
                // InitializeStandarLitSurfaceData initializes based on the rules for standard shader.
                // You can write your own function to initialize the surface data of your shader.
                SurfaceData surfaceData;
                InitializeStandardLitSurfaceData(input.uv, surfaceData);

#if _NORMALMAP
                half3 normalWS = TransformTangentToWorld(surfaceData.normalTS,
                    half3x3(input.tangentWS, input.bitangentWS, input.normalWS));
#else
                half3 normalWS = input.normalWS;
#endif
                normalWS = normalize(normalWS);

#ifdef LIGHTMAP_ON
                // Normal is required in case Directional lightmaps are baked
                half3 bakedGI = SampleLightmap(input.uvLM, normalWS);
#else
                // Samples SH fully per-pixel. SampleSHVertex and SampleSHPixel functions
                // are also defined in case you want to sample some terms per-vertex.
                half3 bakedGI = SampleSH(normalWS);
#endif

                float3 positionWS = input.positionWSAndFogFactor.xyz;
                half3 viewDirectionWS = SafeNormalize(GetCameraPositionWS() - positionWS);

                // BRDFData holds energy conserving diffuse and specular material reflections and its roughness.
                // It's easy to plugin your own shading fuction. You just need replace LightingPhysicallyBased function
                // below with your own.
                BRDFData brdfData;
                InitializeBRDFData(surfaceData.albedo, surfaceData.metallic, surfaceData.specular, surfaceData.smoothness, surfaceData.alpha, brdfData);

                // Light struct is provide by LWRP to abstract light shader variables.
                // It contains light direction, color, distanceAttenuation and shadowAttenuation.
                // LWRP take different shading approaches depending on light and platform.
                // You should never reference light shader variables in your shader, instead use the GetLight
                // funcitons to fill this Light struct.
#ifdef _MAIN_LIGHT_SHADOWS
                // Main light is the brightest directional light.
                // It is shaded outside the light loop and it has a specific set of variables and shading path
                // so we can be as fast as possible in the case when there's only a single directional light
                // You can pass optionally a shadowCoord (computed per-vertex). If so, shadowAttenuation will be
                // computed.
                Light mainLight = GetMainLight(input.shadowCoord);
#else
                Light mainLight = GetMainLight();
#endif

                // Mix diffuse GI with environment reflections.
                half3 color = GlobalIllumination(brdfData, bakedGI, surfaceData.occlusion, normalWS, viewDirectionWS);

                // LightingPhysicallyBased computes direct light contribution.
                color += LightingPhysicallyBased(brdfData, mainLight, normalWS, viewDirectionWS);

                // Additional lights loop
#ifdef _ADDITIONAL_LIGHTS

                // Returns the amount of lights affecting the object being renderer.
                // These lights are culled per-object in the forward renderer
                int additionalLightsCount = GetAdditionalLightsCount();
                for (int i = 0; i < additionalLightsCount; ++i)
                {
                    // Similar to GetMainLight, but it takes a for-loop index. This figures out the
                    // per-object light index and samples the light buffer accordingly to initialized the
                    // Light struct. If _ADDITIONAL_LIGHT_SHADOWS is defined it will also compute shadows.
                    Light light = GetAdditionalLight(i, positionWS);

                    // Same functions used to shade the main light.
                    color += LightingPhysicallyBased(brdfData, light, normalWS, viewDirectionWS);
                }
#endif
                // Emission
                color += surfaceData.emission;

                float fogFactor = input.positionWSAndFogFactor.w;

                // Mix the pixel color with fogColor. You can optionaly use MixFogColor to override the fogColor
                // with a custom one.
                color = MixFog(color, fogFactor);

                // Blend the edges
                float alpha = tex2D(_AlphaMap, input.uvAlpha).x;
                return half4(color, alpha);
                // old return half4(color, surfaceData.alpha);
                //return half4(input.positionCS.z, input.positionCS.z, input.positionCS.z, 1.0);
            }
            ENDHLSL
        }

            // Used for rendering shadowmaps
            UsePass "Universal Render Pipeline/Lit/ShadowCaster"

            // Used for depth prepass
            // If shadows cascade are enabled we need to perform a depth prepass. 
            // We also need to use a depth prepass in some cases camera require depth texture
            // (e.g, MSAA is enabled and we can't resolve with Texture2DMS
            UsePass "Universal Render Pipeline/Lit/DepthOnly"

            // Used for Baking GI. This pass is stripped from build.
            UsePass "Universal Render Pipeline/Lit/Meta"
        }

        // Uses a custom shader GUI to display settings. Re-use the same from Lit shader as they have the
        // same properties.
        //CustomEditor "UnityEditor.Rendering.Universal.ShaderGUI.LitShader"
}