#include "SimManager.h"

#include <imstkCamera.h>
#include <imstkCollisionDetection.h>
#include <imstkCollisionGraph.h>
#include <imstkDynamicObject.h>
#include <imstkLogger.h>
#include <imstkObjectInteractionFactory.h>
#include <imstkScene.h>

#include "DynamicObject.h"

namespace simManager
{
std::shared_ptr<imstk::SceneManager> sdk = nullptr;
std::function<void()> PreUpdate = nullptr;
std::function<void()> PostUpdate = nullptr;

void genSimManager(const int numThreads)
{
    // Create SDK
    sdk = std::make_shared<imstk::SceneManager>();
    // Create the default scene
    sdk->setActiveScene(std::make_shared<imstk::Scene>("DefaultScene"));

    imstk::connect<imstk::Event>(sdk, imstk::EventType::PostUpdate, [&](imstk::Event*) {
        if (PostUpdate != nullptr)
            PostUpdate();
    });
    imstk::connect<imstk::Event>(sdk, imstk::EventType::PreUpdate, [&](imstk::Event*) {
        if (PreUpdate != nullptr)
            PreUpdate();
    });
}

void setSimManagerPostUpdateCallback(void (*func)())
{
    PostUpdate = func;
}

void setSimManagerPreUpdateCallback(void (*func)())
{
    PreUpdate = func;
}

void setWriteTaskGraph(bool writeTaskGraph)
{
    if (sdk != nullptr)
        sdk->getActiveScene()->getConfig()->writeTaskGraph = writeTaskGraph;
}

void addInteractionPair(void* objAHandle,
                        void* objBHandle,
                        int interactionType,
                        int collisionDetectionType)
{
    if (!dynamicObjects.count(objAHandle))
    {
        LOG(INFO) << "Could not find object: " << objAHandle << " for collision pair";
        return;
    }
    std::shared_ptr<imstk::CollidingObject> objA =
        std::dynamic_pointer_cast<imstk::CollidingObject>(dynamicObjects[objAHandle]);
    if (!dynamicObjects.count(objBHandle))
    {
        LOG(INFO) << "Could not find object: " << objBHandle << " for collision pair";
        return;
    }
    std::shared_ptr<imstk::CollidingObject> objB =
        std::dynamic_pointer_cast<imstk::CollidingObject>(dynamicObjects[objBHandle]);

    /* LOG(INFO) << "Collision pair added to scene: (\n\t"
         << "Interaction Type: " << interactionType << "\n\t"
         << "Collision Detection Type: " << collisionDetectionType << "\n\t"
         << "ObjA ID: " << objAInstanceId << "\n\t"
         << "ObjB ID: " << objBInstanceId;*/

    std::shared_ptr<imstk::ObjectInteractionPair> interactionPair =
        imstk::makeObjectInteractionPair(
            objA,
            objB,
            static_cast<imstk::InteractionType>(interactionType),
            static_cast<imstk::CollisionDetection::Type>(collisionDetectionType));

    if (interactionPair == nullptr)
    {
        LOG(INFO) << "Failed to create interaction pair\n";
        LOG(INFO) << "Interaction pair: (\n\t"
                  << "Interaction Type: " << interactionType << "\n\t"
                  << "Collision Detection Type: " << collisionDetectionType << "\n\t"
                  << "ObjA Addr: " << objAHandle << "\n\t"
                  << "ObjB Addr: " << objBHandle << ")\n";
        return;
    }
    else
    {
        LOG(INFO) << "Created interaction pair\n";
        LOG(INFO) << "Interaction pair: (\n\t"
                  << "Interaction Type: " << interactionType << "\n\t"
                  << "Collision Detection Type: " << collisionDetectionType << "\n\t"
                  << "ObjA Addr: " << objAHandle << "\n\t"
                  << "ObjB Addr: " << objBHandle << ")\n";
    }
   

    sdk->getActiveScene()->getCollisionGraph()->addInteraction(interactionPair);
}

void startSimManager()
{
    if (sdk != nullptr)
        sdk->start(false); // Return after starting
    else
        LOG(FATAL) << "Cannot start SimulationManager, does not exist";
}

void initSimManager()
{
    if (sdk != nullptr)
        sdk->init();
    else
        LOG(FATAL) << "Cannot init SimulationManager, does not exist";
}

void advanceFrame(const double dt)
{
    sdk->getActiveScene()->advance(dt);
}

double getActiveSceneDt()
{
    return sdk->getActiveScene()->getElapsedTime();
}

void pauseSimManager()
{
    sdk->pause();
}

void resumeSimManager()
{
    sdk->resume();
}

imstk::ThreadStatus getSimManagerStatus()
{
    return sdk->getStatus();
}

void* getTaskGraph()
{
    if (sdk != nullptr)
    {
        if (sdk->getActiveScene() != nullptr)
        {
            if (sdk->getActiveScene()->getTaskGraph() != nullptr)
            {
                return sdk->getActiveScene()->getTaskGraph().get();
            }
        }
    }
    return nullptr;
}

void stopSimManager()
{
    if (sdk != nullptr)
        sdk->stop(true); // Wait until stopped
    else
        LOG(FATAL) << "Cannot stop SimulationManager, does not exist";
}

void deleteSimManager()
{
    sdk = nullptr;
}
} // namespace simManager